(*********************************************
TThreadTimer -> TComponent

A timer replacement that uses a thread to manage
the timer events.
*********************************************)
unit ThreadTimer;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, MMSystem;

type

  TThreadTimer = class;

  TTimerThread = class( TThread )
  private
    tt: TThreadTimer;
  protected
    procedure DoExecute;
  public
    constructor CreateTimerThread( tt: TThreadTimer );
    procedure Execute; override;
  end;

  TThreadTimer = class(TComponent)
  private
    FInterval: integer;
    FPriority: TThreadPriority;
    FOnTimer: TNotifyEvent;
    bStop: boolean;
    bRunning: boolean;
    FEnabled: boolean;
  protected
    procedure setEnabled( b: boolean );
    procedure Start;
    procedure Stop;
  public
    constructor Create( AOwner: TComponent ); override;
  published
    property Enabled: boolean read FEnabled write setEnabled;
    property Interval: integer read FInterval write FInterval;
    property ThreadPriority: TThreadPriority read FPriority write FPriority default tpNormal;
    property OnTimer: TNotifyEvent read FOnTimer write FOnTimer;
  end;

procedure Register;

implementation

(*********************************************
Initialize thread priority
*********************************************)
constructor TThreadTimer.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FPriority := tpNormal;
end;

(*********************************************
Changing the Enabled property calls either
Start or Stop protected methods.
*********************************************)
procedure TThreadTimer.setEnabled( b: boolean );
begin
  if b then
    Start
  else
    Stop;
  FEnabled := bRunning;
end;

(*********************************************
Starting the timer creates an instance of
TTimerThread and launches the thread.
*********************************************)
procedure TThreadTimer.Start;
begin
  if bRunning then
    Exit;
  bStop := false;
  if not (csDesigning in ComponentState) then
  begin
    with TTimerThread.CreateTimerThread( self ) do
    begin
      Priority := FPriority;
      Resume;
    end;
  end;
  bRunning := true;
end;

(*********************************************
Stopping the timer just sets the stop flag to
true, the TTimerThread's Execute method will
then end and the thread will be destroyed.
*********************************************)
procedure TThreadTimer.Stop;
begin
  bStop := true;
  bRunning := false;
end;

(*********************************************

*********************************************)
constructor TTimerThread.CreateTimerThread( tt: TThreadTimer );
begin
  inherited Create( true );
  self.tt := tt;
  FreeOnTerminate := true;
end;

(*********************************************
Execute method for the spawned thread.  Just
repeats while the timer is enabled, and calls the
timer object's OnTimer event.
*********************************************)
procedure TTimerThread.Execute;
var
  SleepTime, Last: integer;
begin
  while not tt.bStop do
  begin
    Last := timeGetTime;
    Synchronize( DoExecute );
    SleepTime := tt.FInterval - ( timeGetTime - Last );
    if SleepTime < 10 then
      SleepTime := 10;
    sleep( SleepTime );
  end;
end;

(*********************************************
This method is called within the TTimerThread's
Execute, using the Synchronize method.  This is
because we need to call the event handler from
the main VCL thread.
*********************************************)
procedure TTimerThread.DoExecute;
begin
  with tt do
  begin
    if Assigned( FOnTimer ) then
      FOnTimer( tt );
  end;
end;

procedure Register;
begin
  RegisterComponents( 'NonVisual', [TThreadTimer] );
end;

end.
